

abstract type AbstractTest end

struct Test_Homogeneity_circle <: AbstractTest

    np::Integer # Sample size
    B::Integer # Number of bootstrap samples to estimate the characteristic function
    M::Integer # Maximal size of samples used to compute the distribution of distance matrices
    vect_t::Vector{<:Real} # Values of possible coordinates of vectors t at which we compute the caracteristic function
    MC::Integer # Number of Monte Carlo replications to estimate the quantile of the test statistic under H0 (length of vector stats_null_hypothesis)
    radius::Real
    center::Vector{<:Real}
    param_norm::Real
    phi_0::Vector{Vector{ComplexF64}} # value of phi_0 estimated with a very large sample
    stats_null_hypothesis::Vector{<:Real}

end

function Test_Homogeneity_circle(
    np::Integer,
    B::Integer, 
    M::Integer,
    vect_t::Vector{<:Real},
    MC::Integer,
    radius = 1,
    center = [0,1],
    param_norm = 1
)

    @assert np>0 "The sample size argument np should be non negative."
    @assert B>0 "The number of bootstrap samples argument B should be non negative."
    @assert M>0 "The maximal sample size argument M should be non negative."
    @assert MC>0 "The number of Monte Carlo replications to compute the statistic under the null argument MC should be non negative."
    @assert length(vect_t)^Int(M*(M-1)/2) < 100000 "The number of elements in vect_t or M are too large"

    large_sample_circle = generate_regular_grid_circle(10000,radius,center);
    distance_matrix_circle_large_sample = Distances.pairwise(Distances.SqEuclidean(1e-12), large_sample_circle, dims = 2) 
    phi_0 = characteristic_function_boot(distance_matrix_circle_large_sample,B,M,vect_t);

    stats_null_hypothesis = zeros(MC)
    for i in 1:MC
        print(i)
        dist_mat = Distances.pairwise(Distances.SqEuclidean(1e-12), generate_uniform_circle(np,radius,center), dims = 2) 
        hat_phi = characteristic_function_boot(dist_mat,B,M,vect_t);
        stats_null_hypothesis[i] = test_statistic(hat_phi,phi_0)
    end

    stats_null_hypothesis = sort(stats_null_hypothesis)
    
    return Test_Homogeneity_circle(np,B,M,vect_t,MC,radius,center,param_norm,phi_0,stats_null_hypothesis)

end

function test_statistic(
    phi_1,
    phi_2
)
    return mean([mean(abs2.(x)) for x in (phi_1 .- phi_2)])
end

function apply_test(
    test::Test_Homogeneity_circle,
    data::Matrix{<:Real}
)
    dist_mat = Distances.pairwise(Distances.SqEuclidean(1e-12), data, dims = 2) 
    hat_phi = characteristic_function_boot(dist_mat,test.B,test.M,test.vect_t) 
    stat = test_statistic(hat_phi,test.phi_0)
    pval = mean(test.stats_null_hypothesis .> stat)

    return [stat, pval]
end


function Rayleigh_test(
    data
)
    @rput(data)
    R"""
    data_ = t(data)
    pval = unif_test(data = data_, type = "Rayleigh")$p.value
    """
    @rget(pval)
return pval
end

function Bingham_test(
    data
)
    @rput(data)
    R"""
    data_ = t(data)
    pval = unif_test(data = data_, type = "Bingham")$p.value
    """
    @rget(pval)
return pval
end



### Uniformity test from B2025

using LinearAlgebra
using Plots
using Random
using Statistics
using NearestNeighbors

using RCall

import Distributions: Normal, MvNormal, Multinomial, Chisq, quantile, mean, std




abstract type AbstractData end

struct CircleData{T} <: AbstractData

    np::Int
    points::Vector{T}

end


function CircleData(
    np::Integer,
    method::String,
    params...
)

    @assert method in ["uniform","normal","mixture normal", "dynamical system", "Faure 1", "FvM", "MFvM"] "Parameter method should be in ['uniform','normal','mixture normal','dynamical system', 'Faure 1', 'FvM', 'MFvM']"
    if method == "uniform"
        points = random_uniform_circle_data(np)
    elseif method == "FvM"
        points = random_von_Mises_Fischer_circle_data(np, params...)
    elseif method == "MFvM"
        points = random_mixture_von_Mises_Fischer_circle_data(np,params...)
    elseif method == "normal"
        points = random_normal_circle_data(np, params...)
    elseif method == "mixture normal"
        points = random_mixture_normal_circle_data(np,params...)
    elseif method == "dynamical system"
        points = random_dynamical_system_circle_data(np,params...)
    elseif method == "Faure 1"
        points = random_dynamical_system_Faure_1_circle_data(np,params...)
    end

    return CircleData(size(points)[1],points)

end


function random_von_Mises_Fischer_circle_data(
    np::Integer,
    κ::Real,
    center = [1.0,0]
)

    R"""
    library(sphunif);
    #library(foreach);
    #library(doParallel);
    #nCores <- 10;
    #registerDoParallel(nCores);
    """;

    @rput(np)
    kappa = κ
    @rput(kappa)
    @rput(center)

    R"""
    sample = r_alt(n = np, p = 2, alt = "vMF",kappa = kappa, mu = center)[, , 1]
    """
    points = collect(transpose(@rget(sample)))
    angles = acos.(points[1,:]) + 2*(pi .- acos.(points[1,:])).* (points[2,:] .<0)

    return angles
end


function random_mixture_von_Mises_Fischer_circle_data(
    np::Integer,
    κ::Real
)

    R"""
    library(sphunif);
    #library(foreach);
    #library(doParallel);
    #nCores <- 10;
    #registerDoParallel(nCores);
    """;

    @rput(np)
    kappa = κ
    @rput(kappa)
    @rput(center)

    R"""
    sample = r_alt(n = np, p = 2, alt = "MvMF",kappa = kappa)[, , 1]
    """
    points = collect(transpose(@rget(sample)))
    angles = acos.(points[1,:]) + 2*(pi .- acos.(points[1,:])).* (points[2,:] .<0)

    return angles
end


function random_uniform_circle_data(
    np::Integer
)

    points = rand(np)*2*π

    return points
end


function random_dynamical_system_circle_data(
    np::Integer,
    increment::Real,
    index_start=0::Real
)

    points = [(n*increment) % (2*π) for n in index_start:(np+index_start-1)]

    return points
end


function random_normal_circle_data(
    np::Integer,
    mean_::Real,
    std_::Real
)

    norm_dist = Normal(mean_, std_)
    norm_sample = rand(norm_dist,np)
    points = norm_sample.%(2*pi) .+ (norm_sample.<0).*(2*pi)

    return points

end


function random_mixture_normal_circle_data(
    np::Integer,
    mean_::Vector{<:Real},
    std_::Vector{<:Real},
    proba::Vector{<:Real}
)

    @assert size(mean_)==size(std_) "The number of centers and the number of standard deviations parameter should be equal."
    @assert size(mean_)==size(proba) "The number of centers and the number of probabilities should coincide."
    @assert sum(proba.<0)==0 "The probabilities of each component should be non negative."
    @assert sum(proba)==1 "The sum of the probabilitites should be 1."
    n_components = size(mean_)[1]
    n_per_comp = rand(Multinomial(np, proba))
    points = vcat([random_normal_circle_data(n_per_comp[i], mean_[i], std_[i]) for i in 1:n_components]...)
    return points

end

function random_dynamical_system_Faure_1_circle_data(
    np::Integer,
    r::Real
)

    U = 2*π*rand(Float64,1)[1]
    points = [(exp(r*log(2))*(n + U)) % (2*π) for n in 1:np]

    return points
end

function random_dynamical_system_Faure_1_circle_data(
    np::Integer,
    r::Real
)

    U = 2*π*rand(Float64,1)[1]

    points = [(n + U) % (2*π) for n in 1:np]
    for i in 1:ceil(r)
        points2 = copy(points)
        points = (2 .* points2) .% (2*π)
    end

    return points
end



const AbstractDataList = [CircleData]











#--------------------------------------------------------------------------
# Function to Plot data
#--------------------------------------------------------------------------


function plot_data(
    data_circle::CircleData,
    alpha = 0.2
)

    x = [2*π/10000*k for k in 1:10000]
    plot(cos.(x),sin.(x),color = "black",aspect_ratio=:equal, legend = false, xlims = [-1.7,1.7], ylims = [-1.2,1.2],showaxis = false, grid = false)
    plot!(cos.(data_circle.points),sin.(data_circle.points),seriestype=:scatter, alpha = alpha, color = 2)

end


#----------------------------------------------------------
# True Signature
#----------------------------------------------------------

export squared_distance_matrix

function squared_distance_matrix(
    data::CircleData
)

    @assert sum(data.points.<0)==0 "The points should have non-negative coordinates."
    @assert sum(data.points.>2*π)==0 "The points should have coordinates not superior to 2*π."

    return [[sq_distance_circle(x,y) for x in data.points] for y in data.points]

end

function squared_distance_matrix(
    data1::CircleData,
    data2::CircleData
)

    @assert sum(data1.points.<0)==0 "The points should have non-negative coordinates."
    @assert sum(data1.points.>2*π)==0 "The points should have coordinates not superior to 2*π."
    @assert sum(data2.points.<0)==0 "The points should have non-negative coordinates."
    @assert sum(data2.points.>2*π)==0 "The points should have coordinates not superior to 2*π."

    return [[sq_distance_circle(x,y) for x in data1.points] for y in data2.points]

end

function sq_distance_circle(
    x::Float64,
    y::Float64
)
    return min(mod.(abs.(x-y),2*π),2*π .- mod.(abs.(x-y),2*π))^2

end





####################################### Test homogeneity







function compute_mean_vect(
    data_type ::UnionAll,
    sample_size_vect::Vector{<:Integer},
    proportion_nearest_neighbours::Vector{<:Real},
    monte_carlo_size::Integer
)

    mean_vect = Vector{Vector{Matrix{Float64}}}(undef, size(sample_size_vect)[1])
    samples_signatures = Vector{Vector{Matrix{Float64}}}(undef, size(sample_size_vect)[1])
    nthreads = Threads.nthreads()
    n_per_threads = Int.(floor.(ones(nthreads).*monte_carlo_size/nthreads))
    it0 = 1

    while sum(n_per_threads)<monte_carlo_size
        n_per_threads[it0] +=1
        it0 +=1
    end
    result =  Vector{Vector{Matrix{Float64}}}(undef, nthreads)

    
    for i in 1:(size(sample_size_vect)[1])
        print(i)
        sample_size = sample_size_vect[i]
        Threads.@threads for it in 1:nthreads
            result[it] = [debiased_dtm_signature(squared_distance_matrix(data_type(sample_size,"uniform")),proportion_nearest_neighbours) for i in 1:n_per_threads[it]]
        end
        dtm_sigs = vcat(result...)
        v   = [hcat([sort(dtm_sig[j,:]) for dtm_sig in dtm_sigs]...) for j in 1:size(proportion_nearest_neighbours)[1]] # All signatures are sorted
        mean_dtm_sig = [mean(v[j],dims = 2) for j in 1:size(proportion_nearest_neighbours)[1]]
        mean_vect[i] = mean_dtm_sig
        samples_signatures[i] = v
    end

    return [mean_vect, samples_signatures]

end



struct Test_Homogeneity <: AbstractTest

    np::Integer
    data_type::UnionAll
    proportion_nearest_neighbours::Vector{<:Real}
    monte_carlo_size::Integer
    barycenter_signatures::Vector{Matrix{<:Real}}
    true_signatures::Vector{<:Real}
    samples_signatures::Vector{<:Vector{<:Matrix{<:Real}}}
    stats_null_hypothesis_homogeneity::Vector{<:Vector{<:Real}}
    stats_null_hypothesis_iidness::Vector{<:Vector{<:Real}}

end

function Test_Homogeneity(
    np::Integer,
    proportion_nearest_neighbours::Vector{<:Real},
    monte_carlo_size::Integer,
    data_type::UnionAll
)

    @assert np>0 "The sample size argument np should be non negative."
    @assert data_type in AbstractDataList "The data_type should belong to [CircleData,SphereData,TorusData,BolzaData]."
    @assert all((proportion_nearest_neighbours.>0) .&& (proportion_nearest_neighbours.<=1)) "proportion_nearest_neighbours is a vector with elements in (0,1]."
    @assert monte_carlo_size>0 "monte_carlo_size argument should be non negative."

    barycenter_signatures, samples_signatures = compute_mean_vect(data_type,[np],proportion_nearest_neighbours,monte_carlo_size)

    barycenter_signatures = barycenter_signatures[1]
    true_signatures = compute_true_signatures(data_type,proportion_nearest_neighbours)

    n_params = size(proportion_nearest_neighbours)[1]

    stats_null_hypothesis_homogeneity = [sort(distance_signatures_test_homogeneity(samples_signatures[1][j], true_signatures[j],np)) for j in 1:n_params]
    stats_null_hypothesis_iidness = [sort(distance_signatures_test_iidness(samples_signatures[1][j], barycenter_signatures[j],np)) for j in 1:n_params]

    
    return Test_Homogeneity(np,data_type,proportion_nearest_neighbours,monte_carlo_size,barycenter_signatures,true_signatures,samples_signatures,stats_null_hypothesis_homogeneity,stats_null_hypothesis_iidness)

end




export apply_test

for datatype in [:CircleData]

    @eval function apply_test(
        t::Test_Homogeneity,
        data::$datatype,
        method::String
    ) 

        @assert method in ["homogeneity","iidness"] "The method is either 'homogeneity' or 'iidness'."
        @assert t.data_type == $datatype "The data type should be the same as the data type for the test."
        @assert t.np == data.np "The sample data should be of the same size as the test method."

        signatures = debiased_dtm_signature(squared_distance_matrix(data),t.proportion_nearest_neighbours)
        p_value = [0]

        n_params = size(t.proportion_nearest_neighbours)[1]
        statistic = zeros(n_params)
        p_value = zeros(n_params)

        if method == "homogeneity"
            for j in 1:n_params
                statistic[j] = distance_signatures_test_homogeneity(signatures[j,:], t.true_signatures[j],t.np)
                p_value[j] = mean([stat >= statistic[j] for stat in t.stats_null_hypothesis_homogeneity[j]])
            end
        elseif method == "iidness"
            for j in 1:n_params
                statistic[j] = distance_signatures_test_iidness(sort(signatures[j,:]), t.barycenter_signatures[j],t.np)
                p_value[j] = mean([stat >= statistic[j] for stat in t.stats_null_hypothesis_iidness[j]])
            end
        end

        return [p_value, statistic]

    end

end


function compute_mean_vect(
    data_type ::UnionAll,
    sample_size_vect::Vector{<:Integer},
    proportion_nearest_neighbours::Vector{<:Real},
    monte_carlo_size::Integer,
    test_0::Test_Homogeneity
)

    n_params = size(proportion_nearest_neighbours)[1]
    monte_carlo_size = monte_carlo_size*n_params

    mean_vect = Vector{Vector{Matrix{Float64}}}(undef, size(sample_size_vect)[1])
    samples_signatures_homogeneity = Vector{Vector{Matrix{Float64}}}(undef, size(sample_size_vect)[1])
    samples_signatures_iidness = Vector{Vector{Matrix{Float64}}}(undef, size(sample_size_vect)[1])

    nthreads = Threads.nthreads()
    n_per_threads = Int.(floor.(ones(nthreads).*monte_carlo_size/nthreads))
    it0 = 1

    while sum(n_per_threads)<monte_carlo_size
        n_per_threads[it0] +=1
        it0 +=1
    end
    result =  Vector{Vector{Matrix{Float64}}}(undef, nthreads)
    indices_min_pval_homogeneity = Vector{Vector{Integer}}(undef, nthreads)
    indices_min_pval_iidness = Vector{Vector{Integer}}(undef, nthreads)
    
    for i in 1:(size(sample_size_vect)[1])
        sample_size = sample_size_vect[i]
        Threads.@threads for it in 1:nthreads
            result[it] = Vector{Matrix{Float64}}(undef,n_per_threads[it])
            indices_min_pval_homogeneity[it] = Vector{Integer}(undef,n_per_threads[it])
            indices_min_pval_iidness[it] = Vector{Integer}(undef,n_per_threads[it])
            for j in 1:n_per_threads[it]
                sample = data_type(sample_size,"uniform")
                index_min_pval_homogeneity = argmin(apply_test(test_0,sample,"homogeneity")[1])
                index_min_pval_iidness = argmin(apply_test(test_0,sample,"iidness")[1])
                indices_min_pval_homogeneity[it][j] = index_min_pval_homogeneity
                indices_min_pval_iidness[it][j] = index_min_pval_iidness
                result[it][j] = debiased_dtm_signature(squared_distance_matrix(sample),proportion_nearest_neighbours)
            end
        end

        indices_homogeneity = vcat(indices_min_pval_homogeneity...)
        indices_iidness = vcat(indices_min_pval_iidness...)
        dtm_sigs = vcat(result...)


        v   = [hcat([sort(dtm_sigs[i][j,:]) for i in 1:size(dtm_sigs)[1] if indices_iidness[i] == j]...) for j in 1:size(proportion_nearest_neighbours)[1]] # All signatures are sorted
        mean_dtm_sig = [mean(v[j],dims = 2) for j in 1:size(proportion_nearest_neighbours)[1]]
        mean_vect[i] = mean_dtm_sig
        samples_signatures_iidness[i] = v

        w   = [hcat([sort(dtm_sigs[i][j,:]) for i in 1:size(dtm_sigs)[1] if indices_homogeneity[i] == j]...) for j in 1:size(proportion_nearest_neighbours)[1]] 
        samples_signatures_homogeneity[i] = w

    end

    return [mean_vect, samples_signatures_homogeneity, samples_signatures_iidness]

end

function debiased_dtm_signature(
    squared_distances_matrix::Vector{<:Vector{<:Real}},
    proportion_nearest_neighbours::Vector{<:Real}
)

    np = size(squared_distances_matrix)[1]

    indices = 1:np
    indicator_vector = [(indices .<= floor( h*(np-1) + 1)) + ((h*(np-1))%1)* (indices .== ceil( h*(np-1) + 1)) for h in proportion_nearest_neighbours]
    dtm_sig = vcat([hcat([sqrt.(1/(proportion_nearest_neighbours[j]*(np-1))*sum( (sort(squared_distances_matrix[i])).*indicator_vector[j] )) for i in 1:np]...) for j in 1:size(proportion_nearest_neighbours)[1]]...)

    return dtm_sig

end



function compute_true_signatures(
    data_type::UnionAll,
    proportion_nearest_neighbours::Vector{<:Real}
)

    @assert data_type in AbstractDataList "The data_type should belong to [CircleData,SphereData,TorusData,BolzaData]."

    if data_type == CircleData
        true_signatures = get_true_signatures_circle(proportion_nearest_neighbours)
    end


return true_signatures

end


function get_true_signatures_circle(
    proportion_nearest_neighbours::Vector{<:Real}
)

    [1/sqrt(3)*π*h for h in proportion_nearest_neighbours]

end


function distance_signatures_test_homogeneity(
    signatures::Matrix{<:Real},
    true_signature::Real,
    np::Integer
)

    distance = 1/np*sum(abs.(signatures .- true_signature), dims = 1)

    return  distance[1,:]

end

function distance_signatures_test_homogeneity(
    signatures::Vector{<:Real},
    true_signature::Real,
    np::Integer
)

    return 1/np*sum(abs.(signatures .- true_signature))

end

function distance_signatures_test_iidness(
    signatures::Matrix{<:Real},
    barycenter_signatures::Matrix{<:Real},
    np::Integer
)

    distance = sqrt.(1/np.*sum((signatures .- barycenter_signatures).^2, dims = 1))
    return distance[1,:]

end

function distance_signatures_test_iidness(
    signatures::Vector{<:Real},
    barycenter_signatures::Matrix{<:Real},
    np::Integer
)

    return sqrt(1/np*sum((signatures.- barycenter_signatures).^2))

end


struct Test_Homogeneity_Param_Selection <: AbstractTest

    np::Integer
    data_type::UnionAll
    proportion_nearest_neighbours::Vector{<:Real}
    monte_carlo_size::Integer
    barycenter_signatures::Vector{Matrix{<:Real}}
    true_signatures::Vector{<:Real}
    samples_signatures_homogeneity::Vector{<:Vector{<:Matrix{<:Real}}}
    samples_signatures_iidness::Vector{<:Vector{<:Matrix{<:Real}}}
    stats_null_hypothesis_homogeneity::Vector{<:Vector{<:Real}}
    stats_null_hypothesis_iidness::Vector{<:Vector{<:Real}}
    test_0::Test_Homogeneity

end

function Test_Homogeneity_Param_Selection(
    np::Integer,
    proportion_nearest_neighbours::Vector{<:Real},
    monte_carlo_size::Integer,
    data_type::UnionAll
)

    @assert np>0 "The sample size argument np should be non negative."
    @assert data_type in AbstractDataList "The data_type should belong to [CircleData,SphereData,TorusData,BolzaData]."
    @assert all((proportion_nearest_neighbours.>0) .&& (proportion_nearest_neighbours.<=1)) "proportion_nearest_neighbours is a vector with elements in (0,1]."
    @assert monte_carlo_size>0 "monte_carlo_size argument should be non negative."

    test_0 = Test_Homogeneity(np,proportion_nearest_neighbours,monte_carlo_size,data_type)

    barycenter_signatures, samples_signatures_homogeneity, samples_signatures_iidness = compute_mean_vect(data_type,[np],proportion_nearest_neighbours,monte_carlo_size,test_0)

    barycenter_signatures = barycenter_signatures[1]
    true_signatures = compute_true_signatures(data_type,proportion_nearest_neighbours)

    n_params = size(proportion_nearest_neighbours)[1]

    stats_null_hypothesis_homogeneity = [sort(distance_signatures_test_homogeneity(samples_signatures_homogeneity[1][j], true_signatures[j],np)) for j in 1:n_params]
    stats_null_hypothesis_iidness = [sort(distance_signatures_test_iidness(samples_signatures_iidness[1][j], barycenter_signatures[j],np)) for j in 1:n_params]

    
    return Test_Homogeneity_Param_Selection(np,data_type,proportion_nearest_neighbours,monte_carlo_size,barycenter_signatures,true_signatures,samples_signatures_homogeneity,samples_signatures_iidness,stats_null_hypothesis_homogeneity,stats_null_hypothesis_iidness,test_0)

end




export apply_test

for datatype in [:CircleData] 

    @eval function apply_test(
        t::Test_Homogeneity_Param_Selection,
        data::$datatype,
        method::String
    ) 

        @assert method in ["homogeneity","iidness"] "The method is either 'homogeneity' or 'iidness'."
        @assert t.data_type == $datatype "The data type should be the same as the data type for the test."
        @assert t.np == data.np "The sample data should be of the same size as the test method."

        if method == "homogeneity"
            index = argmin(apply_test(t.test_0, data, "homogeneity")[1])
        elseif method == "iidness"
            index = argmin(apply_test(t.test_0, data,"iidness")[1])
        end

        signatures = debiased_dtm_signature(squared_distance_matrix(data),[t.proportion_nearest_neighbours[index]])

        statistic = 0
        p_value = 0

        if method == "homogeneity"
            statistic = distance_signatures_test_homogeneity(signatures[1,:], t.true_signatures[index],t.np)
            p_value = mean([stat >= statistic for stat in t.stats_null_hypothesis_homogeneity[index]])
        elseif method == "iidness"
            statistic = distance_signatures_test_iidness(sort(signatures[1,:]), t.barycenter_signatures[index],t.np)
            p_value = mean([stat >= statistic for stat in t.stats_null_hypothesis_iidness[index]])
        end

        return [p_value, statistic, index, t.proportion_nearest_neighbours[index]]

    end

end




struct Test_Homogeneity_Aggregated <: AbstractTest

    np::Integer
    data_type::UnionAll
    proportion_nearest_neighbours::Vector{<:Real}
    weights::Vector{<:Real}
    monte_carlo_size::Integer
    monte_carlo_size_2::Integer
    vect_alpha::Vector{<:Real}
    vect_quantiles_homogeneity::Vector{Vector{<:Real}}
    vect_quantiles_iidness::Vector{Vector{<:Real}}
    test_0::Test_Homogeneity

end


function Test_Homogeneity_Aggregated(
    np::Integer,
    proportion_nearest_neighbours::Vector{<:Real},
    weights::Vector{<:Real},
    monte_carlo_size::Integer,
    monte_carlo_size_2::Integer,
    data_type::UnionAll,
    vect_alpha::Vector{<:Real},
    error_approx_u = 0.001
)

    @assert np>0 "The sample size argument np should be non negative."
    @assert data_type in AbstractDataList "The data_type should belong to [CircleData,SphereData,TorusData,BolzaData]."
    @assert all((proportion_nearest_neighbours.>0) .&& (proportion_nearest_neighbours.<=1)) "proportion_nearest_neighbours is a vector with elements in (0,1]."
    @assert monte_carlo_size>0 "monte_carlo_size argument should be non negative."
    @assert size(weights)[1] == size(proportion_nearest_neighbours)[1] "The vector weights should have the same length as the vector proportion_nearest_neighbours."

    test_0 = Test_Homogeneity(np,proportion_nearest_neighbours,monte_carlo_size,data_type)

    n_params = size(proportion_nearest_neighbours)[1]

        

    statistics_2_homogeneity = [Vector{Float64}(undef,monte_carlo_size_2) for _ in 1:n_params]
    statistics_2_iidness = [Vector{Float64}(undef,monte_carlo_size_2) for _ in 1:n_params]

    for n_ in 1:monte_carlo_size_2
        data = data_type(np,"uniform")
        signatures = debiased_dtm_signature(squared_distance_matrix(data),proportion_nearest_neighbours)

        statistic = zeros(n_params)

        for j in 1:n_params
            statistics_2_homogeneity[j][n_] = distance_signatures_test_homogeneity(signatures[j,:], test_0.true_signatures[j],np)
        end

        for j in 1:n_params
            statistics_2_iidness[j][n_] = distance_signatures_test_iidness(sort(signatures[j,:]), test_0.barycenter_signatures[j],np)
        end

    end

    statistics_2 = [statistics_2_homogeneity, statistics_2_iidness]


    quant_type = eltype(test_0.stats_null_hypothesis_homogeneity[1])

    vect_quantiles_homogeneity = Vector{Vector{quant_type}}(undef,size(vect_alpha)[1])
    vect_quantiles_iidness = Vector{Vector{quant_type}}(undef,size(vect_alpha)[1])

    for (idx_alpha, alpha) in enumerate(vect_alpha)
        quantiles = Vector{Vector{quant_type}}(undef,2)

        for (idx_,statistics) in enumerate([test_0.stats_null_hypothesis_homogeneity, test_0.stats_null_hypothesis_iidness])

            Δ = 1
            u_min = 0
            u_max = 1
            u = mean([u_min,u_max])
            while Δ > error_approx_u
                centered_stats = hcat([ statistics_2[idx_][idx] .- stat[Int(ceil(monte_carlo_size*(1 - u*weights[idx])))] for (idx,stat) in enumerate(statistics)]...) 
                T = mean(maximum(centered_stats, dims = 2) .> 0)
                if T > alpha
                    u_max = u
                else
                    u_min = u
                end
                Δ = u_max - u_min
                u = mean([u_min,u_max])
            end

            quantiles[idx_] = [stat[Int(ceil(monte_carlo_size*(1 - u*weights[idx])))] for (idx, stat) in enumerate(statistics)]
        end
        
        vect_quantiles_homogeneity[idx_alpha] = quantiles[1]
        vect_quantiles_iidness[idx_alpha] = quantiles[2]
    end

    return Test_Homogeneity_Aggregated(np,data_type,proportion_nearest_neighbours,weights,monte_carlo_size,monte_carlo_size_2,vect_alpha,vect_quantiles_homogeneity,vect_quantiles_iidness,test_0)

end


export apply_test

for datatype in [:CircleData]

    @eval function apply_test(
        t::Test_Homogeneity_Aggregated,
        data::$datatype,
        method::String
    ) 

        @assert method in ["homogeneity","iidness"] "The method is either 'homogeneity' or 'iidness'."
        @assert t.data_type == $datatype "The data type should be the same as the data type for the test."
        @assert t.np == data.np "The sample data should be of the same size as the test method."
   
        signatures = debiased_dtm_signature(squared_distance_matrix(data),t.proportion_nearest_neighbours)

        n_params = size(t.proportion_nearest_neighbours)[1]

        statistic = zeros(n_params)

        vect_stat = Vector{eltype(t.vect_quantiles_homogeneity[1])}(undef,size(t.vect_alpha)[1])
        vect_param_max = Vector{eltype(t.proportion_nearest_neighbours)}(undef,size(t.vect_alpha)[1])

        if method == "homogeneity"
            for j in 1:n_params
                statistic[j] = distance_signatures_test_homogeneity(signatures[j,:], t.test_0.true_signatures[j],t.test_0.np)
            end
            for (idx_alpha, quant_hom) in enumerate(t.vect_quantiles_homogeneity)
                diff_quantile = [st - quant_hom[idx] for (idx, st) in enumerate(statistic)]
                stat, index_max = findmax(diff_quantile)
                param_max = t.proportion_nearest_neighbours[index_max]
                vect_stat[idx_alpha] = stat
                vect_param_max[idx_alpha] = param_max
            end
        elseif method == "iidness"
            for j in 1:n_params
                statistic[j] = distance_signatures_test_iidness(sort(signatures[j,:]), t.test_0.barycenter_signatures[j],t.test_0.np)
            end
            for (idx_alpha, quant_iid) in enumerate(t.vect_quantiles_iidness)
                diff_quantile = [st - quant_iid[idx] for (idx, st) in enumerate(statistic)]
                stat, index_max = findmax(diff_quantile)
                param_max = t.proportion_nearest_neighbours[index_max]
                vect_stat[idx_alpha] = stat
                vect_param_max[idx_alpha] = param_max
            end
        end

        return [vect_stat, vect_param_max]

    end

end
